<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require "../../connect.php";

// Make sure we have a request method
$requestMethod = $_SERVER['REQUEST_METHOD'] ?? 'GET';

/**
 * Fallback helper functions in case they are not defined elsewhere
 */
if (!function_exists('error')) {
    function error($msg) {
        return json_encode([
            'status'  => 0,
            'message' => $msg,
        ], JSON_UNESCAPED_UNICODE);
    }
}
if (!function_exists('success')) {
    function success($msg) {
        return json_encode([
            'status'  => 1,
            'message' => $msg,
        ], JSON_UNESCAPED_UNICODE);
    }
}
if (!function_exists('serverError')) {
    function serverError() {
        return json_encode([
            'status'  => 0,
            'message' => 'Server error',
        ], JSON_UNESCAPED_UNICODE);
    }
}
if (!function_exists('error405')) {
    function error405($method) {
        return json_encode([
            'status'  => 0,
            'message' => 'Method not allowed: ' . $method,
        ], JSON_UNESCAPED_UNICODE);
    }
}

// Route handler
if ($requestMethod === "POST") {
    echo editProduct();
} else {
    echo error405($requestMethod);
}

function editProduct() {
    global $con;

    $headers = array_change_key_case(getallheaders(), CASE_LOWER);
    $lang = strtolower(trim($headers['lang'] ?? ''));

    // multipart/form-data → use $_POST + $_FILES
    $obj = $_POST;

    /* ======================= BASIC CHECKS ======================= */
    if (!isset($obj['product_id']) || empty(trim($obj['product_id']))) {
        return error($lang === 'en' ? 'Enter product ID' : 'ادخل معرف المنتج');
    }

    $product_id = (int) mysqli_real_escape_string($con, $obj['product_id']);

    // Check product exists
    $sql_select = mysqli_query($con, "SELECT * FROM `products` WHERE `product_id` = '$product_id' LIMIT 1");
    if (!$sql_select || mysqli_num_rows($sql_select) < 1) {
        return error($lang === 'en' ? 'This product does not exist' : 'هذا المنتج غير موجود');
    }
    $product_row = mysqli_fetch_object($sql_select);

    /* ======================= BUILD UPDATE LIST (PARTIAL) ======================= */
    $updates = [];
    $didSomething = false;

    // product_name (AR)
    if (isset($obj['product_name'])) {
        if (empty(trim($obj['product_name']))) {
            return error($lang === 'en' ? 'Enter product name in Arabic' : 'ادخل اسم المنتج بالعربي');
        }
        $product_name = mysqli_real_escape_string($con, $obj['product_name']);
        $updates[] = "`product_name` = '$product_name'";
    }

    // product_name_en
    if (isset($obj['product_name_en'])) {
        if (empty(trim($obj['product_name_en']))) {
            return error($lang === 'en' ? 'Enter product name in English' : 'ادخل اسم المنتج بالانجليزي');
        }
        $product_name_en = mysqli_real_escape_string($con, $obj['product_name_en']);
        $updates[] = "`product_name_en` = '$product_name_en'";
    }

    // product_description (AR)
    if (isset($obj['product_description'])) {
        if (empty(trim($obj['product_description']))) {
            return error($lang === 'en' ? 'Enter product description in Arabic' : 'ادخل وصف المنتج بالعربي');
        }
        $product_description = mysqli_real_escape_string($con, $obj['product_description']);
        $updates[] = "`product_description` = '$product_description'";
    }

    // product_description_en
    if (isset($obj['product_description_en'])) {
        if (empty(trim($obj['product_description_en']))) {
            return error($lang === 'en' ? 'Enter product description in English' : 'ادخل وصف المنتج بالانجليزي');
        }
        $product_description_en = mysqli_real_escape_string($con, $obj['product_description_en']);
        $updates[] = "`product_description_en` = '$product_description_en'";
    }

    // product_url
    if (isset($obj['product_url'])) {
        $product_url = trim($obj['product_url']);
        if ($product_url === '') {
            return error($lang === 'en' ? 'Enter product URL' : 'ادخل رابط المنتج');
        }
        $product_url = mysqli_real_escape_string($con, $product_url);
        $updates[]   = "`product_url` = '$product_url'";
    }

    /* ======================= PRICE TYPE + PRICE ======================= */
    if (isset($obj['price_type'])) {
        if (empty(trim($obj['price_type']))) {
            return error($lang === 'en' ? 'Enter price type' : 'ادخل نوع السعر');
        }

        $new_price_type = mysqli_real_escape_string($con, $obj['price_type']);
        $updates[]      = "`price_type` = '$new_price_type'";

        if ($new_price_type === "free") {
            $updates[] = "`product_price` = '0'";
        } else {
            if (isset($obj['product_price'])) {
                if (empty(trim($obj['product_price']))) {
                    return error($lang === 'en' ? 'Enter product price' : 'ادخل سعر المنتج');
                }
                $product_price = mysqli_real_escape_string($con, $obj['product_price']);
                $updates[]     = "`product_price` = '$product_price'";
            } else {
                if ($product_row->price_type === "free") {
                    return error($lang === 'en' ? 'Enter product price' : 'ادخل سعر المنتج');
                }
            }
        }
    } else {
        if (isset($obj['product_price'])) {
            if (empty(trim($obj['product_price']))) {
                return error($lang === 'en' ? 'Enter product price' : 'ادخل سعر المنتج');
            }
            $product_price = mysqli_real_escape_string($con, $obj['product_price']);
            $updates[]     = "`product_price` = '$product_price'";
        }
    }

    /* ======================= CATEGORIES ONLY (NO SUBCATEGORY) ======================= */
    if (isset($obj['cat_id'])) {
        if (empty(trim($obj['cat_id']))) {
            return error($lang === 'en' ? 'Enter product categories ids' : 'ادخل معرفات الأقسام التابع اليها المنتج');
        }
        $cat_id = mysqli_real_escape_string($con, $obj['cat_id']);
        $cats   = explode(",", $cat_id);

        foreach ($cats as $cat) {
            $cat = trim($cat);
            if ($cat === '') continue;

            $sql_cat = mysqli_query($con, "SELECT * FROM `categories` WHERE `cat_id` = '$cat' AND `hidden` = '0'");
            if (!$sql_cat || mysqli_num_rows($sql_cat) < 1) {
                return error($lang === 'en' ? 'This category ' . $cat . ' does not exist' : 'هذا القسم ' . $cat . ' غير موجود');
            }
        }

        $updates[] = "`product_catId` = '$cat_id'";
    }

    /* ======================= COLORS (UPDATE products.product_colors) ======================= */
    // Accept:
    // - colors as JSON string: '["#fff","#000"]' or '[{"hex":"#fff"}]'
    // - colors[] in form-data
    if (array_key_exists('colors', $obj)) {
        $colorsRaw = $obj['colors'];

        if (is_string($colorsRaw)) {
            $decoded = json_decode($colorsRaw, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $colors = $decoded;
            } else {
                // If string but not valid JSON, treat as single value
                $colors = [$colorsRaw];
            }
        } elseif (is_array($colorsRaw)) {
            $colors = $colorsRaw;
        } else {
            $colors = [];
        }

        // Normalize (remove empty values)
        $cleanColors = [];
        foreach ($colors as $c) {
            if (is_string($c)) {
                $v = trim($c);
                if ($v !== '') $cleanColors[] = $v;
            } elseif (is_array($c)) {
                // keep object-like entries if your frontend uses them
                $cleanColors[] = $c;
            }
        }

        // Save JSON or NULL
        if (count($cleanColors) === 0) {
            $updates[] = "`product_colors` = NULL";
        } else {
            $colorsJson = mysqli_real_escape_string($con, json_encode($cleanColors, JSON_UNESCAPED_UNICODE));
            $updates[] = "`product_colors` = '$colorsJson'";
        }

        $didSomething = true;
    }

    /* ======================= SIZES (UPDATE product_props) ======================= */
    // Accept:
    // - sizes as JSON string: [{"title":"Small","price":0},{"title":"Large","price":10}]
    // - sizes as array in form-data
    // If sizes is sent:
    // - delete old rows
    // - insert new rows
    $sizesProvided = array_key_exists('sizes', $obj);
    $sizes = null;

    if ($sizesProvided) {
        $sizesRaw = $obj['sizes'];

        if (is_string($sizesRaw)) {
            $decoded = json_decode($sizesRaw, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $sizes = $decoded;
            } else {
                // If invalid JSON, sizes will be treated as empty
                $sizes = [];
            }
        } elseif (is_array($sizesRaw)) {
            $sizes = $sizesRaw;
        } else {
            $sizes = [];
        }

        // Clear old sizes
        mysqli_query($con, "DELETE FROM `product_props` WHERE `product_id` = '$product_id'");

        // Insert new sizes
        if (is_array($sizes) && count($sizes) > 0) {
            foreach ($sizes as $size) {
                // support both keys: price or value
                $title = '';
                $price = '';

                if (is_array($size)) {
                    $title = $size['title'] ?? ($size['name'] ?? '');
                    $price = $size['price'] ?? ($size['value'] ?? 0);
                }

                $title = mysqli_real_escape_string($con, (string)$title);
                $price = mysqli_real_escape_string($con, (string)$price);

                if (trim($title) === '') {
                    continue;
                }

                mysqli_query(
                    $con,
                    "INSERT INTO `product_props` (`title`, `value`, `product_id`) VALUES ('$title', '$price', '$product_id')"
                );
            }
        }

        $didSomething = true;
    }

    /* ======================= FILE UPLOAD SETUP ======================= */
    $uploadDir     = __DIR__ . '/../uploads/products/';
    $uploadUrlBase = 'uploads/products/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }

    $allowedExt = ['jpg', 'jpeg', 'png', 'webp'];

    /* ======================= MAIN PRODUCT IMAGE ======================= */
    if (isset($_FILES['product_image']) && $_FILES['product_image']['error'] === UPLOAD_ERR_OK) {
        $mainTmpName  = $_FILES['product_image']['tmp_name'];
        $mainOrigName = $_FILES['product_image']['name'];
        $mainExt      = strtolower(pathinfo($mainOrigName, PATHINFO_EXTENSION));

        if (!in_array($mainExt, $allowedExt, true)) {
            return error($lang === 'en' ? 'Invalid main image type' : 'نوع الصورة الرئيسية غير مدعوم');
        }

        $mainFileName = uniqid('prod_main_') . '.' . $mainExt;
        $mainDest     = $uploadDir . $mainFileName;

        if (!move_uploaded_file($mainTmpName, $mainDest)) {
            return serverError();
        }

        // Delete old main image file if it exists and is local
        $oldMainUrl = $product_row->product_image ?? '';
        if (!empty($oldMainUrl)) {
            $oldMainPath = __DIR__ . '/../' . ltrim($oldMainUrl, '/');
            if (file_exists($oldMainPath)) {
                @unlink($oldMainPath);
            }
        }

        $product_image = $uploadUrlBase . $mainFileName;
        $updates[]     = "`product_image` = '$product_image'";
        $didSomething  = true;
    }

    /* ======================= GALLERY: DELETE IMAGES NOT IN existing_images[] ======================= */
    $existingImagesRaw = $obj['existing_images'] ?? ($obj['exiting_images'] ?? null);

    if ($existingImagesRaw !== null) {
        if (is_string($existingImagesRaw)) {
            $decoded = json_decode($existingImagesRaw, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $existingImagesRaw = $decoded;
            } else {
                $existingImagesRaw = [$existingImagesRaw];
            }
        }

        $normalizedExisting = [];

        if (is_array($existingImagesRaw)) {
            foreach ($existingImagesRaw as $imgVal) {
                if (!is_string($imgVal) || $imgVal === '') {
                    continue;
                }

                $imgPath = trim($imgVal);

                if (strpos($imgPath, 'http') === 0) {
                    $pathPart = parse_url($imgPath, PHP_URL_PATH);
                    if (!$pathPart) {
                        continue;
                    }
                    $pos = strpos($pathPart, '/uploads/');
                    if ($pos !== false) {
                        $imgPath = ltrim(substr($pathPart, $pos), '/');
                    } else {
                        $normalizedExisting[] = $imgPath;
                        continue;
                    }
                }

                $imgPath = ltrim($imgPath, '/');
                if ($imgPath !== '') {
                    $normalizedExisting[] = $imgPath;
                }
            }
        }

        $sql_imgs = mysqli_query(
            $con,
            "SELECT `id`, `url` FROM `product_images` WHERE `product_id` = '$product_id'"
        );

        if ($sql_imgs) {
            while ($imgRow = mysqli_fetch_assoc($sql_imgs)) {
                $dbUrl = ltrim($imgRow['url'], '/');

                if (!in_array($dbUrl, $normalizedExisting, true)) {
                    $fullPath = __DIR__ . '/../' . $dbUrl;
                    if (file_exists($fullPath)) {
                        @unlink($fullPath);
                    }
                    $img_id = (int)$imgRow['id'];
                    mysqli_query(
                        $con,
                        "DELETE FROM `product_images` WHERE `id` = '$img_id' AND `product_id` = '$product_id'"
                    );
                    $didSomething = true;
                }
            }
        }
    }

    /* ======================= APPLY UPDATES (PRODUCT TABLE) ======================= */
    if (!empty($updates)) {
        $setClause = implode(", ", $updates);
        $sql_update = mysqli_query(
            $con,
            "UPDATE `products` SET $setClause WHERE `product_id` = '$product_id'"
        );
        if (!$sql_update) {
            return serverError();
        }
        $didSomething = true;
    }

    /* ======================= ADD NEW GALLERY IMAGES ======================= */
    if (isset($_FILES['images']) && is_array($_FILES['images']['name'])) {
        $countImages = count($_FILES['images']['name']);
        for ($i = 0; $i < $countImages; $i++) {
            if ($_FILES['images']['error'][$i] !== UPLOAD_ERR_OK) {
                continue;
            }

            $tmpName  = $_FILES['images']['tmp_name'][$i];
            $origName = $_FILES['images']['name'][$i];
            $ext      = strtolower(pathinfo($origName, PATHINFO_EXTENSION));

            if (!in_array($ext, $allowedExt, true)) {
                continue;
            }

            $fileName = uniqid('prod_img_') . '.' . $ext;
            $dest     = $uploadDir . $fileName;

            if (move_uploaded_file($tmpName, $dest)) {
                $imageUrl = $uploadUrlBase . $fileName;
                mysqli_query(
                    $con,
                    "INSERT INTO `product_images`(`url`, `product_id`) VALUES ('$imageUrl','$product_id')"
                );
                $didSomething = true;
            }
        }
    }

    /* ======================= NOTHING UPDATED ======================= */
    if (!$didSomething) {
        return error($lang === 'en' ? 'No data to update' : 'لا يوجد بيانات لتعديلها');
    }

    return success($lang === 'en' ? 'Product updated successfully' : 'تم تعديل المنتج بنجاح');
}
