<?php
// send-notification.php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *'); // Adjust to specific origins in production
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

ini_set('display_errors', 0); // Disable in production
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

require "../connect.php";

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

// FCM Server Key (use environment variable or config file in production)
$fcmServerKey = getenv('FCM_SERVER_KEY') ?: 'YOUR_FCM_SERVER_KEY_HERE';

try {
    // Get POST data
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    // Validate required fields
    if (!isset($data['user_id']) || !isset($data['title']) || !isset($data['body'])) {
        throw new Exception('Missing required fields: user_id, title, body');
    }

    $user_id = $data['user_id'];
    $title = htmlspecialchars($data['title'], ENT_QUOTES, 'UTF-8');
    $body = htmlspecialchars($data['body'], ENT_QUOTES, 'UTF-8');
    $fcm_token = 'SELECT fcm_token FROM `users` WHERE user_id = "$user_id"';

    // Validate FCM token (basic check)
    if (empty($fcm_token)) {
        throw new Exception('Invalid FCM token');
    }

    // Prepare FCM payload
    $fcmPayload = [
        'to' => $fcm_token,
        'notification' => [
            'title' => $title,
            'body' => $body,
            'sound' => 'default',
            'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
        ],
        'data' => [
            'timestamp' => time(),
            'sent_from_server' => 'true'
        ],
        'android' => [
            'priority' => 'high',
            'notification' => [
                'channel_id' => 'chat_messages',
                'sound' => 'default',
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                'icon' => 'ic_launcher',
            ]
        ],
        'apns' => [
            'headers' => [
                'apns-priority' => '10'
            ],
            'payload' => [
                'aps' => [
                    'alert' => [
                        'title' => $title,
                        'body' => $body
                    ],
                    'sound' => 'default',
                    'badge' => 1
                ]
            ]
        ]
    ];

    // Prepare HTTP context for POST request
    $options = [
        'http' => [
            'method' => 'POST',
            'header' => [
                'Authorization: key=' . $fcmServerKey,
                'Content-Type: application/json'
            ],
            'content' => json_encode($fcmPayload),
            'timeout' => 30
        ],
        'ssl' => [
            'verify_peer' => true,
            'verify_peer_name' => true
        ]
    ];
    $context = stream_context_create($options);

    // Send notification to FCM
    $response = @file_get_contents('https://fcm.googleapis.com/fcm/send', false, $context);

    // Check for request errors
    if ($response === false) {
        $error = error_get_last();
        throw new Exception('HTTP Request Error: ' . ($error['message'] ?? 'Unknown error'));
    }

    // Get HTTP status code
    $httpCode = 0;
    if (isset($http_response_header[0])) {
        preg_match('/\d{3}/', $http_response_header[0], $matches);
        $httpCode = (int)($matches[0] ?? 0);
    }

    // Parse FCM response
    $fcmResponse = json_decode($response, true);

    // Check FCM response
    if ($httpCode !== 200) {
        $errorMessage = isset($fcmResponse['error']) ? $fcmResponse['error'] : 'Unknown FCM error';
        throw new Exception('FCM Error (HTTP ' . $httpCode . '): ' . $errorMessage);
    }

    // Check for FCM-specific errors
    if (isset($fcmResponse['failure']) && $fcmResponse['failure'] > 0) {
        $errorDetails = isset($fcmResponse['results'][0]['error']) ? 
            $fcmResponse['results'][0]['error'] : 'Unknown failure';
        throw new Exception('FCM Send Failure: ' . $errorDetails);
    }

    // Log success
    logNotification($data, true);

    // Success response
    echo json_encode([
        'success' => true,
        'message' => 'Notification sent successfully',
        'fcm_response' => $fcmResponse,
        'timestamp' => date('Y-m-d H:i:s')
    ]);

} catch (Exception $e) {
    // Log error
    logNotification($data ?? [], false, $e->getMessage());

    // Error response
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}

// Log notification attempt
function logNotification($data, $success, $error = null) {
    $logEntry = [
        'timestamp' => date('Y-m-d H:i:s'),
        // 'fcm_token' => substr($data['fcm_token'] ?? '', 0, 20) . '...',
        'title' => $data['title'] ?? '',
        'success' => $success,
        'error' => $error
    ];
    
    file_put_contents(
        'notification_log.txt', 
        json_encode($logEntry) . "\n", 
        FILE_APPEND | LOCK_EX
    );
}
?>