<?php
require '../connect.php';

ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__.'/order-error.log');
error_reporting(E_ALL);

function jsonResponse($data, $code = 200) {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}
function apiError($msg, $code = 400) {
    jsonResponse(['success' => false, 'message' => $msg], $code);
}
function apiSuccess($data = [], $msg = 'Success') {
    jsonResponse(['success' => true, 'message' => $msg, 'data' => $data]);
}


$input = file_get_contents('php://input');
if (!$input) apiError('Empty request');
$req = json_decode($input, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    apiError('Invalid JSON');
}


$action = $req['action'];

function createOrder(){
global $jsonResponse, $apiError, $apiSuccess;
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    apiError('Use POST', 405);
}

$input = file_get_contents('php://input');
if (!$input) apiError('Empty request');
$req = json_decode($input, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    apiError('Invalid JSON');
}

// === Extract Data ===
$user_id           = trim($req['user_id'] ?? '');
$address_id        = (int)($req['address_id'] ?? 0);
$phone_number      = trim($req['phone_number'] ?? '');
$receiver_phone    = trim($req['receiver_phone_number'] ?? '');
$receiver_address  = trim($req['receiver_address'] ?? '');
$orders            = $req['orders'] ?? [];
$total             = (float)($req['total'] ?? 0);
$shipping_cost     = (float)($req['shipping_cost'] ?? 0);

// === Validate ===
if (!$user_id)           apiError('user_id is required');
if (!$address_id)        apiError('address_id is required');
if (!$phone_number)      apiError('Your phone is required');
if (!$receiver_phone)    apiError('Receiver phone is required');
if (!$receiver_address)  apiError('Receiver address is required');
if (empty($orders))      apiError('No items in cart');

// === DB Connect ===
require '../connect.php';
if (!$con || $con->connect_error) {
    apiError('Database error', 500);
}

// === Validate Address (SAFE) ===
$address_valid = false;
try {
    $stmt = $con->prepare("SELECT 1 FROM addresses WHERE address_id = ? AND user_id = ? LIMIT 1");
    if ($stmt) {
        $stmt->bind_param('is', $address_id, $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $address_valid = $res->num_rows > 0;
        $stmt->close();
    }
} catch (Throwable $e) {
    error_log("Address check failed: " . $e->getMessage());
}

if (!$address_valid) {
    apiError('Invalid or unauthorized address (or table missing)');
}

// === Calculate Subtotal ===
$subtotal = 0;
foreach ($orders as $item) {
    $price = (float)($item['unit_after_disc'] ?? 0);
    $qty   = (int)($item['quantity'] ?? 0);
    $subtotal += $price * $qty;
}

$calculated = $subtotal + $shipping_cost;
if (abs($calculated - $total) > 0.01) {
    apiError("Total mismatch. Expected: $calculated, got: $total");
}

// === Transaction ===
$con->autocommit(false);
try {
    $stmt = $con->prepare("
        INSERT INTO orders (user_id, address_id, shipping_cost, total, created_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    $stmt->bind_param('sidd', $user_id, $address_id, $shipping_cost, $total);
    $stmt->execute();
    $order_id = $con->insert_id;
    $stmt->close();

    $stmtItem = $con->prepare("
        INSERT INTO order_items (order_id, product_id, price, quantity)
        VALUES (?, ?, ?, ?)
    ");
    foreach ($orders as $item) {
        $pid = (int)$item['product_id'];
        $price = (float)$item['unit_after_disc'];
        $qty = (int)$item['quantity'];
        $stmtItem->bind_param('iidi', $order_id, $pid, $price, $qty);
        $stmtItem->execute();
    }
    $stmtItem->close();

    $gift_msg = trim(($req['message'] ?? '') . ' ' . ($req['bouquet_phrase'] ?? ''));
    if ($gift_msg || $receiver_phone || $receiver_address) {
        $stmtGift = $con->prepare("
            INSERT INTO order_gifts 
            (order_id, recipient_name, recipient_phone, gift_message, receiver_address, additional_info)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        if ($stmtGift) {
            $name = 'Customer';
            $info = trim($req['address_info'] ?? '');
            $stmtGift->bind_param('isssss', $order_id, $name, $receiver_phone, $gift_msg, $receiver_address, $info);
            $stmtGift->execute();
            $stmtGift->close();
        }
    }

    $stmtClear = $con->prepare("DELETE FROM cart WHERE item_userId = ?");
    $stmtClear->bind_param('s', $user_id);
    $stmtClear->execute();
    $stmtClear->close();

    $con->commit();
} catch (Throwable $e) {
    $con->rollback();
    apiError('Order failed: ' . $e->getMessage(), 500);
}
$con->autocommit(true);

apiSuccess(['order_id' => $order_id, 'total' => $total], 'Order created!');
}


function listOrders(){
global $jsonResponse, $apiError, $apiSuccess, $req;

$user_id = trim($req['user_id'] ?? '');
$lang    = in_array($req['lang'] ?? '', ['ar','en']) ? $req['lang'] : 'ar';

if (!$user_id) {
    apiError($lang === 'ar' ? 'معرف المستخدم مطلوب' : 'user_id is required');
}

require __DIR__.'/../connect.php';
if (!$con || $con->connect_error) {
    apiError('Database error', 500);
}

$orders = [];
$stmt = $con->prepare("
    SELECT 
        o.order_id,
        o.user_id,
        o.address_id,
        o.shipping_cost,
        o.total,
        o.status,
        o.created_at,
        a.full_address,
        g.recipient_phone AS address_phone,
        g.recipient_name,
        g.recipient_phone,
        g.gift_message,
        g.receiver_address,
        g.additional_info
    FROM orders o
    LEFT JOIN (
        SELECT address_id, 
               CONCAT(street, ', ', building_number, ', ', gover) AS full_address
        FROM addresses
    ) a ON a.address_id = o.address_id
    LEFT JOIN order_gifts g ON g.order_id = o.order_id
    WHERE o.user_id = ?
    ORDER BY o.created_at DESC
");
$stmt->bind_param('s', $user_id);
$stmt->execute();
$res = $stmt->get_result();

while ($order = $res->fetch_assoc()) {
    $order_id = $order['order_id'];

    $items = [];
    $itemStmt = $con->prepare("
        SELECT oi.product_id, oi.price, oi.quantity,
               p.product_name" . ($lang === 'en' ? '_en' : '') . " AS name,
              pi.*
        FROM order_items oi
        JOIN products p ON p.product_id = oi.product_id
        LEFT JOIN product_images pi ON p.product_id = pi.product_id
        WHERE oi.order_id = ?
    ");
    $itemStmt->bind_param('i', $order_id);
    $itemStmt->execute();
    $itemRes = $itemStmt->get_result();

    while ($item = $itemRes->fetch_assoc()) {
        $items[] =$item;
    }
    $itemStmt->close();

    $orders[] = [
        'order_id'        => (int)$order_id,
        'total'           => (float)$order['total'],
        'shipping_cost'   => (float)$order['shipping_cost'],
        'status'          => $order['status'] ?? 'pending',
        'created_at'      => $order['created_at'],
        'address'         => $order['full_address'] ? [
            'full_address' => $order['full_address'],
            'phone'        => $order['address_phone']
        ] : null,
        'gift'            => $order['gift_message'] ? [
            'recipient_name'     => $order['recipient_name'],
            'recipient_phone'    => $order['recipient_phone'],
            'gift_message'       => $order['gift_message'],
            'receiver_address'   => $order['receiver_address'],
            'additional_info'    => $order['additional_info']
        ] : null,
        'items'           => $items
    ];
}
$stmt->close();

apiSuccess([
    'total_orders' => count($orders),
    'orders'       => $orders
], $lang === 'ar' ? 'تم جلب الطلبات بنجاح' : 'Orders loaded successfully');
}


function reOrder(){
global $jsonResponse, $apiError, $apiSuccess, $req;

$user_id    = trim($req['user_id'] ?? '');
$order_id   = (int)($req['order_id'] ?? 0);
$lang       = in_array($req['lang'] ?? '', ['ar','en']) ? $req['lang'] : 'ar';

if (!$user_id)  apiError($lang === 'ar' ? 'معرف المستخدم مطلوب' : 'user_id is required');
if (!$order_id) apiError($lang === 'ar' ? 'معرف الطلب مطلوب' : 'order_id is required');

require __DIR__.'/../connect.php';
if (!$con || $con->connect_error) {
    apiError('Database error', 500);
}

// === 5. Fetch Original Order ===
$stmt = $con->prepare("
    SELECT o.address_id, o.shipping_cost, o.total,
           g.recipient_name, g.recipient_phone, g.gift_message, g.receiver_address, g.additional_info
    FROM orders o
    LEFT JOIN addresses a ON a.address_id = o.address_id
    LEFT JOIN order_gifts g ON g.order_id = o.order_id
    WHERE o.order_id = ? AND o.user_id = ?
");
$stmt->bind_param('is', $order_id, $user_id);
$stmt->execute();
$order = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$order) {
    apiError($lang === 'ar' ? 'الطلب غير موجود أو غير مملوك لك' : 'Order not found or not yours');
}

// === 6. Fetch Order Items ===
$items = [];
$stmt = $con->prepare("
    SELECT oi.product_id, oi.price, oi.quantity
    FROM order_items oi
    WHERE oi.order_id = ?
");
$stmt->bind_param('i', $order_id);
$stmt->execute();
$res = $stmt->get_result();

while ($row = $res->fetch_assoc()) {
    $items[] = [
        'product_id'      => (int)$row['product_id'],
        'unit_after_disc' => (float)$row['price'],
        'quantity'        => (int)$row['quantity']
    ];
}
$stmt->close();

if (empty($items)) {
    apiError($lang === 'ar' ? 'لا توجد عناصر في الطلب' : 'No items in order');
}

// === 7. Re-calculate Total ===
$subtotal = 0;
foreach ($items as $item) {
    $subtotal += $item['unit_after_disc'] * $item['quantity'];
}
$total = $subtotal + (float)$order['shipping_cost'];

// === 8. Start Transaction ===
$con->autocommit(false);
try {
    // Insert New Order
    $stmt = $con->prepare("
        INSERT INTO orders (user_id, address_id, shipping_cost, total, created_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    $stmt->bind_param('sidd', $user_id, $order['address_id'], $order['shipping_cost'], $total);
    $stmt->execute();
    $new_order_id = $con->insert_id;
    $stmt->close();

    // Insert Items
    $stmtItem = $con->prepare("
        INSERT INTO order_items (order_id, product_id, price, quantity)
        VALUES (?, ?, ?, ?)
    ");
    foreach ($items as $item) {
        $stmtItem->bind_param('iidi', $new_order_id, $item['product_id'], $item['unit_after_disc'], $item['quantity']);
        $stmtItem->execute();
    }
    $stmtItem->close();

    // Copy Gift Note
    $gift_msg = trim(($order['gift_message'] ?? ''));
    if ($gift_msg || $order['recipient_phone'] || $order['receiver_address']) {
        $stmtGift = $con->prepare("
            INSERT INTO order_gifts 
            (order_id, recipient_name, recipient_phone, gift_message, receiver_address, additional_info)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        if ($stmtGift) {
            $name = $order['recipient_name'] ?? 'Customer';
            $info = $order['additional_info'] ?? '';
            $stmtGift->bind_param('isssss', $new_order_id, $name, $order['recipient_phone'], $gift_msg, $order['receiver_address'], $info);
            $stmtGift->execute();
            $stmtGift->close();
        }
    }

    $con->commit();
} catch (Throwable $e) {
    $con->rollback();
    apiError('Reorder failed: ' . $e->getMessage(), 500);
}
$con->autocommit(true);

// === 9. SUCCESS ===
apiSuccess([
    'original_order_id' => $order_id,
    'new_order_id'      => $new_order_id,
    'total'             => round($total, 2)
], $lang === 'ar' ? 'تم إعادة الطلب بنجاح!' : 'Order recreated successfully!');
}

if($action == "add"){
    createOrder();
}else if($action == "list"){
    listOrders();
}else if($action == "reorder"){
    reOrder();
}else{
   apiError('Invalid Action'); 
}


