<?php

ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__.'/favorites-error.log');
error_reporting(E_ALL);

function jsonResponse(array $data, int $code = 200) {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}
function apiError(string $msg, int $code = 400) {
    jsonResponse(['success'=>false, 'message'=>$msg], $code);
}
function apiSuccess($payload = [], string $msg = 'Success') {
    jsonResponse(['success'=>true, 'message'=>$msg, 'data'=>$payload]);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    apiError('Use POST', 405);
}

$input = file_get_contents('php://input');
if ($input === false) apiError('Cannot read input');
$data = json_decode($input, true);
if (json_last_error() !== JSON_ERROR_NONE) apiError('Invalid JSON: '.json_last_error_msg());

$action  = strtolower(trim($data['action'] ?? ''));
$user_id = trim($data['user_id'] ?? '');
if ($action === '')  apiError('action is required');
if ($user_id === '') apiError('user_id is required');

require __DIR__.'/../connect.php';
if (!$con || $con->connect_error) {
    error_log('DB connect: '.$con->connect_error);
    apiError('Database unavailable', 500);
}


$stmtCheck = $con->prepare("SELECT favorite_id FROM favorites WHERE favorite_userId = ? AND favorite_productId = ? LIMIT 1");
$stmtIns   = $con->prepare("INSERT INTO favorites (favorite_userId, favorite_productId) VALUES (?, ?)");
$stmtDel   = $con->prepare("DELETE FROM favorites WHERE favorite_userId = ? AND favorite_productId = ?");

if (!$stmtCheck || !$stmtIns || !$stmtDel) {
    $err = $con->error;
    error_log("Prepare failed: $err");
    apiError("Prepare failed: $err", 500);
}


$product_id = (int)($data['product_id'] ?? 0);
if (($action === 'add' || $action === 'remove') && $product_id <= 0) {
    apiError("product_id required for '$action'");
}


if ($action === 'add') {
    $con->autocommit(false);
    try {
        $stmtCheck->bind_param('si', $user_id, $product_id);
        $stmtCheck->execute();
        $stmtCheck->store_result();
        if ($stmtCheck->num_rows === 0) {
            $stmtIns->bind_param('si', $user_id, $product_id);
            if (!$stmtIns->execute()) {
                throw new Exception("Insert failed: " . $stmtIns->error);
            }
        } else {
            
        }
        $con->commit();
    } catch (Throwable $e) {
        $con->rollback();
        $errorMsg = $e->getMessage();
        error_log("ADD ERROR: $errorMsg");
        apiError("Add failed: $errorMsg", 500);
    }
    $con->autocommit(true);
} elseif ($action === 'remove') {
    $stmtDel->bind_param('si', $user_id, $product_id);
    if (!$stmtDel->execute()) {
        apiError("Remove failed: " . $stmtDel->error, 500);
    }
} elseif ($action === 'clear') {
    $stmt = $con->prepare("DELETE FROM favorites WHERE favorite_userId = ?");
    if (!$stmt) apiError("Clear prepare failed: " . $con->error, 500);
    $stmt->bind_param('s', $user_id);
    $stmt->execute();
    $stmt->close();
} elseif (!in_array($action, ['list', 'select'])) {
    apiError("Invalid action: $action");
}


$favorites = []; $total_favorites = 0;

$q = "
    SELECT f.favorite_id, f.favorite_productId, f.favorite_userId,
           p.product_name, p.product_price, p.product_discount, p.product_image
    FROM favorites f
    LEFT JOIN products p ON p.product_id = f.favorite_productId
    WHERE f.favorite_userId = ?
    ORDER BY f.favorite_id DESC
";
$stmt = $con->prepare($q);
if (!$stmt) {
    error_log("SELECT prepare failed: " . $con->error);
    apiError("SELECT failed: " . $con->error, 500);
}
$stmt->bind_param('s', $user_id);
$stmt->execute();
$res = $stmt->get_result();

while ($row = $res->fetch_assoc()) {
    $price = (float)($row['product_price'] ?? 0);
    $disc  = (float)($row['product_discount'] ?? 0);
    $after = max(0, $price - $disc);

    $favorites[] = [
        'favorite_id'      => (int)$row['favorite_id'],
        'product_id'       => (int)$row['favorite_productId'],
        'product_name'     => $row['product_name'] ?? 'Unknown',
        'product_image'    => $row['product_image'] ?? '',
        'unit_price'       => $price,
        'discount'         => $disc,
        'unit_after_disc'  => $after,
    ];
    $total_favorites++;
}
$stmt->close();

apiSuccess([
    'user_id'          => $user_id,
    'action'           => $action,
    'total_favorites'  => $total_favorites,
    'favorites'        => $favorites
], 'Favorites processed');